/**
 * Manages the rendering and cleanup of components with caching and lifecycle hooks
 */
export class ComponentLoader {
  constructor() {
    this.components = {};
    this.targets = new Map(); // Cache DOM elements
    this.cleanups = new Map(); // Store cleanup functions per target
    this.DEBUG = (import.meta.env?.MODE ?? 'production') !== 'production';
  }

  /**
   * Registers a component with optional lifecycle hooks
   * @param {string} name - The component name
   * @param {Object} component - An object with render, init, and cleanup functions
   */
  register(name, component) {
    this.components[name] = {
      render: component.render || (() => {}),
      init: component.init || (() => {}),
      cleanup: component.cleanup || (() => {}),
    };
  }

  /**
   * Renders a component into a target element with caching
   * Calls cleanup first to remove previous listeners, then stores new cleanup if returned
   * @param {string} name - The component name
   * @param {string} targetId - The ID of the element to render into
   */
  render(name, targetId) {
    const component = this.components[name];
    if (!component) {
      if (this.DEBUG) console.warn(`Component ${name} not found.`);
      return;
    }
    let target = this.targets.get(targetId);
    if (!target) {
      target = document.getElementById(targetId);
      if (target) this.targets.set(targetId, target);
      else {
        if (this.DEBUG) console.warn(`Target ${targetId} not found.`);
        return;
      }
    }

    // Call existing cleanup if it exists
    const existingCleanup = this.cleanups.get(targetId);
    if (existingCleanup) {
      existingCleanup();
      this.cleanups.delete(targetId);
    }

    // Initialize and render, capturing any returned cleanup function
    component.init(target);
    const newCleanup = component.render(target);

    // Store the cleanup function if it’s returned
    if (typeof newCleanup === 'function') {
      this.cleanups.set(targetId, newCleanup);
    }
  }

  /**
   * Clears cached targets and cleanups (for a full reset)
   */
  clearCache() {
    this.cleanups.forEach((cleanup) => cleanup());
    this.cleanups.clear();
    this.targets.clear();
  }
}