/**
 * Renders the meta navigation into the provided target element.
 * If no target is provided, it uses the element with id "meta-nav".
 * @param {HTMLElement} [target] - The element to render the meta navigation into.
 * @param {string} [lang='de'] - The language code (e.g., 'de' or 'en').
 */
export function renderMetaNav(target, lang = 'de') {
  const container = target || document.getElementById('meta-nav');
  if (!container) return;

  // Clear existing content.
  container.innerHTML = '';

  // --- Language Selector ---
  const langContainer = document.createElement('div');
  langContainer.id = 'lang-container';
  langContainer.className = 'relative flex items-center';

  const langBtn = document.createElement('button');
  langBtn.dataset.action = 'toggle-lang';
  langBtn.className = 'flex items-center gap-2 hover:text-gray-600 focus:outline-none';
  langBtn.setAttribute('aria-expanded', 'false');
  langBtn.setAttribute('aria-controls', 'lang-dropdown');

  const langImg = document.createElement('img');
  langImg.src = '/assets/icons/language.svg';
  langImg.alt = 'Language Selector';
  langImg.className = 'w-6 h-6';
  langImg.loading = 'lazy';

  const langText = document.createElement('span');
  langText.className = 'hidden md:inline nav-link';
  langText.textContent = lang === 'de' ? 'Deutsch' : 'English';

  const arrowSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  arrowSvg.classList.add('hidden', 'md:inline', 'w-4', 'h-4');
  arrowSvg.setAttribute('fill', 'currentColor');
  arrowSvg.setAttribute('viewBox', '0 0 20 20');
  const arrowPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
  arrowPath.setAttribute('fill-rule', 'evenodd');
  arrowPath.setAttribute('d', 'M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z');
  arrowPath.setAttribute('clip-rule', 'evenodd');
  arrowSvg.appendChild(arrowPath);

  langBtn.append(langImg, langText, arrowSvg);

  const langDropdown = document.createElement('div');
  langDropdown.id = 'lang-dropdown';
  langDropdown.setAttribute('role', 'menu');
  langDropdown.className = 'absolute right-0 top-full mt-2 w-32 bg-white border border-gray-200 rounded shadow-md transition-all duration-200 origin-top scale-0 opacity-0 pointer-events-none';
  langDropdown.setAttribute('aria-hidden', 'true');

  const langUl = document.createElement('ul');
  langUl.className = 'py-1 text-sm';
  const languages = [
    { code: 'de', label: 'Deutsch' },
    { code: 'en', label: 'English' }
  ];
  languages.forEach(item => {
    const li = document.createElement('li');
    const a = document.createElement('a');
    a.href = '#';
    a.setAttribute('role', 'menuitem');
    a.setAttribute('aria-selected', (item.code === lang).toString());
    a.className = 'block px-4 py-2 hover:bg-gray-50';
    a.textContent = item.label;
    a.addEventListener('click', (e) => {
      e.preventDefault();
      langText.textContent = item.label;
      // No actual language switching as per requirement
    });
    li.appendChild(a);
    langUl.appendChild(li);
  });
  langDropdown.appendChild(langUl);

  langContainer.appendChild(langBtn);
  langContainer.appendChild(langDropdown);

  // Divider
  const divider = document.createElement('div');
  divider.className = 'hidden md:block w-px h-5 bg-gray-300 ml-3';

  // --- Search Container ---
  const searchContainer = document.createElement('div');
  searchContainer.id = 'search-container';
  searchContainer.className = 'relative flex items-center';

  const searchBtn = document.createElement('button');
  searchBtn.dataset.action = 'toggle-search';
  searchBtn.className = 'flex items-center gap-2 hover:text-gray-600 focus:outline-none';
  searchBtn.setAttribute('aria-expanded', 'false');
  searchBtn.setAttribute('aria-controls', 'search-field');
  searchBtn.id = 'search-btn';

  const searchIcon = document.createElement('img');
  searchIcon.src = '/assets/icons/search.svg';
  searchIcon.alt = 'Search';
  searchIcon.className = 'w-4 h-4';
  searchIcon.loading = 'lazy';

  const searchSpan = document.createElement('span');
  searchSpan.className = 'hidden md:inline nav-link';
  searchSpan.textContent = 'Suche';

  const searchShortcut = document.createElement('img');
  searchShortcut.src = '/assets/images/search-shortcut.svg';
  searchShortcut.alt = 'Cmd + K';
  searchShortcut.className = 'hidden md:inline w-8 h-8';
  searchShortcut.loading = 'lazy';

  searchBtn.append(searchIcon, searchSpan, searchShortcut);

  const searchField = document.createElement('div');
  searchField.id = 'search-field';
  searchField.className = 'hidden md:absolute top-full right-0 mt-2 bg-white border border-gray-300 rounded px-3 py-2 w-48 transition-all duration-200 origin-top scale-0 opacity-0 pointer-events-none';
  searchField.setAttribute('aria-hidden', 'true');

  const searchLabel = document.createElement('label');
  searchLabel.htmlFor = 'search-input';
  searchLabel.className = 'sr-only';
  searchLabel.textContent = 'Search';

  const searchInput = document.createElement('input');
  searchInput.type = 'search';
  searchInput.id = 'search-input';
  searchInput.placeholder = 'Search...';
  searchInput.className = 'w-full focus:outline-none focus:ring-2 focus:ring-blue-500';

  searchField.appendChild(searchLabel);
  searchField.appendChild(searchInput);

  const liveRegion = document.createElement('div');
  liveRegion.id = 'search-live-region';
  liveRegion.className = 'sr-only';
  liveRegion.setAttribute('aria-live', 'polite');

  searchContainer.append(searchBtn, searchField, liveRegion);

  // --- Register Section ---
  const registerContainer = document.createElement('div');
  registerContainer.className = 'flex items-center gap-2';

  const registerIcon = document.createElement('img');
  registerIcon.src = '/assets/icons/register.svg';
  registerIcon.alt = 'Register';
  registerIcon.className = 'w-6 h-6';
  registerIcon.loading = 'lazy';

  const registerLink = document.createElement('a');
  registerLink.href = '#';
  registerLink.className = 'hidden md:inline nav-link';
  registerLink.textContent = 'Anmelden';

  registerContainer.append(registerIcon, registerLink);

  // --- Mobile Menu Toggle ---
  const mobileToggle = document.createElement('button');
  mobileToggle.dataset.action = 'toggle-mobile';
  mobileToggle.className = 'md:hidden p-3 rounded text-gray-600 focus:outline-none';
  mobileToggle.setAttribute('aria-expanded', 'false');
  mobileToggle.setAttribute('aria-controls', 'mobile-menu');

  const mobileSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  mobileSvg.classList.add('w-6', 'h-6');
  mobileSvg.setAttribute('fill', 'none');
  mobileSvg.setAttribute('stroke', 'currentColor');
  mobileSvg.setAttribute('stroke-width', '2');
  mobileSvg.setAttribute('viewBox', '0 0 24 24');
  mobileSvg.innerHTML = '<path stroke-linecap="round" stroke-linejoin="round" d="M4 6h16M4 12h16M4 18h16"></path>';
  mobileToggle.appendChild(mobileSvg);

  // Append all elements in order.
  container.appendChild(langContainer);
  container.appendChild(divider);
  container.appendChild(searchContainer);
  container.appendChild(registerContainer);
  container.appendChild(mobileToggle);

  // Add Event Listener for Search Input.
  const searchListener = (e) => {
    liveRegion.textContent = `Searching for ${e.target.value}`;
  };
  searchInput.addEventListener('input', searchListener);

  // Store cleanup function.
  container.__cleanupMetaNav = () => {
    searchInput.removeEventListener('input', searchListener);
  };
}

/**
 * Renders the main navigation into the provided target element.
 * If no target is provided, it uses the element with id "main-nav".
 * @param {HTMLElement} [target] - The element to render the main navigation into.
 */
export function renderMainNav(target) {
  const container = target || document.getElementById('main-nav');
  if (!container) return;
  container.innerHTML = '';

  const items = ['Produkt', 'Bestellen', 'Download', 'Ressourcen', 'Karriere', 'Unternehmen'];

  const navWrapper = document.createElement('div');
  navWrapper.className = 'bg-brandGray rounded-2xl px-4 py-4 flex items-center justify-between';

  const ul = document.createElement('ul');
  ul.className = 'flex items-center space-x-8';

  items.forEach(item => {
    const li = document.createElement('li');
    const a = document.createElement('a');
    a.href = '#';
    a.className = 'nav-link';
    a.textContent = item;
    li.appendChild(a);
    ul.appendChild(li);
  });
  navWrapper.appendChild(ul);

  const ctaContainer = document.createElement('div');
  ctaContainer.className = 'flex items-center';
  const ctaLink = document.createElement('a');
  ctaLink.href = '#';
  ctaLink.className = 'btn-cta-nav';
  ctaLink.textContent = 'Jetzt kostenlos testen';
  ctaContainer.appendChild(ctaLink);

  navWrapper.appendChild(ctaContainer);
  container.appendChild(navWrapper);
}

/**
 * Renders the mobile menu into the provided target element.
 * If no target is provided, it uses the element with id "mobile-menu".
 * @param {HTMLElement} [target] - The element to render the mobile menu into.
 */
export function renderMobileMenu(target) {
  const container = target || document.getElementById('mobile-menu');
  if (!container) return;
  container.innerHTML = '';

  const items = ['Produkt', 'Bestellen', 'Download', 'Ressourcen', 'Karriere', 'Unternehmen'];

  const closeBtn = document.createElement('button');
  closeBtn.dataset.action = 'toggle-mobile';
  closeBtn.className = 'absolute top-4 right-4 text-gray-600 hover:text-gray-800 focus:outline-none';
  closeBtn.setAttribute('aria-label', 'Close Menu');
  const closeSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  closeSvg.classList.add('w-6', 'h-6');
  closeSvg.setAttribute('fill', 'none');
  closeSvg.setAttribute('stroke', 'currentColor');
  closeSvg.setAttribute('stroke-width', '2');
  closeSvg.setAttribute('viewBox', '0 0 24 24');
  closeSvg.innerHTML = '<path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"></path>';
  closeBtn.appendChild(closeSvg);

  const ul = document.createElement('ul');
  ul.className = 'h-full flex flex-col items-center justify-center space-y-6 text-center w-full';
  ul.setAttribute('aria-label', 'Mobile Navigation');

  items.forEach(item => {
    const li = document.createElement('li');
    li.className = 'w-full';
    const a = document.createElement('a');
    a.href = '#';
    a.className = 'mobile-nav-link';
    a.textContent = item;
    li.appendChild(a);
    ul.appendChild(li);
  });

  const ctaLi = document.createElement('li');
  ctaLi.className = 'w-full';
  const ctaLink = document.createElement('a');
  ctaLink.href = '#';
  ctaLink.className = 'btn-cta-mobile';
  ctaLink.textContent = 'Jetzt kostenlos testen';
  ctaLi.appendChild(ctaLink);
  ul.appendChild(ctaLi);

  container.appendChild(closeBtn);
  container.appendChild(ul);

  // Automatically close the mobile menu when a link is clicked.
  container.querySelectorAll('a').forEach(link => {
    link.addEventListener('click', () => {
      import('./../utils/domUtils.js').then(({ toggleOverlay }) => {
        toggleOverlay({ id: 'mobile-menu', iframeId: null, open: false });
      });
    });
  });
}

/**
 * Renders the mobile search overlay into the provided target element.
 * If no target is provided, it uses the element with id "mobile-search".
 * @param {HTMLElement} [target] - The element to render the mobile search into.
 */
export function renderMobileSearch(target) {
  const container = target || document.getElementById('mobile-search');
  if (!container) return;
  container.innerHTML = '';

  const searchOverlay = document.createElement('div');
  searchOverlay.className = 'relative bg-white w-full max-w-sm p-4 rounded';

  const closeBtn = document.createElement('button');
  closeBtn.dataset.action = 'close-search';
  closeBtn.className = 'absolute -top-6 -right-5 w-8 h-8 flex items-center justify-center bg-white text-gray-600 hover:bg-gray-200 rounded-full shadow-md border border-gray-300 focus:outline-none';
  closeBtn.setAttribute('aria-label', 'Close Search');
  const closeSvg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  closeSvg.classList.add('w-4', 'h-4');
  closeSvg.setAttribute('fill', 'none');
  closeSvg.setAttribute('stroke', 'currentColor');
  closeSvg.setAttribute('stroke-width', '2');
  closeSvg.setAttribute('viewBox', '0 0 24 24');
  closeSvg.innerHTML = '<path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"></path>';
  closeBtn.appendChild(closeSvg);

  const searchInput = document.createElement('input');
  searchInput.type = 'search';
  searchInput.placeholder = 'Search...';
  searchInput.className = 'w-full border border-gray-300 rounded p-2 focus:outline-none focus:ring-2 focus:ring-blue-500 mt-2';
  searchInput.id = 'mobile-search-input';

  searchOverlay.appendChild(closeBtn);
  searchOverlay.appendChild(searchInput);
  container.appendChild(searchOverlay);
}