/**
 * Renders the video modal into the provided target element using DOM APIs.
 * Ensures that when closed the modal is removed so that reopening always creates a new modal.
 * @param {HTMLElement} target - The container element to render the video modal into.
 * @param {string} [lang='en'] - The language code (e.g., 'en' or 'de').
 * @param {string} [videoUrl='https://www.youtube.com/embed/8KGN6e0OnDY?modestbranding=1&rel=0&autoplay=1']
 * @returns {Function} A cleanup function to remove event listeners.
 */
export function renderVideoModal(target, lang = 'en', videoUrl = 'https://www.youtube.com/embed/8KGN6e0OnDY?modestbranding=1&rel=0&autoplay=1') {
  if (!target) {
    console.warn('Target element for VideoModal not found.');
    return () => {};
  }

  // Basic translations
  const translations = {
    en: {
      closeLabel: 'Close Video',
      videoTitle: 'think-cell Product Demo'
    },
    de: {
      closeLabel: 'Video schließen',
      videoTitle: 'think-cell Produktdemo'
    }
  };
  const t = translations[lang] || translations.en;

  // Save the element that was focused before opening the modal.
  const previouslyFocused = document.activeElement;

  // Clear any existing content in the target container and ensure it’s visible.
  target.innerHTML = '';
  target.classList.remove('hidden');

  // Create modal container.
  const modalDiv = document.createElement('div');
  modalDiv.className = 'bg-white rounded p-4 relative w-11/12 max-w-4xl transition-opacity duration-200 opacity-0';
  modalDiv.setAttribute('role', 'dialog');
  modalDiv.setAttribute('aria-modal', 'true');
  modalDiv.setAttribute('aria-labelledby', 'videoModalTitle');
  modalDiv.setAttribute('aria-describedby', 'videoModalDesc');

  // Create close button.
  const closeBtn = document.createElement('button');
  closeBtn.dataset.action = 'close-video';
  closeBtn.className = 'absolute -top-6 -right-4 w-8 h-8 flex items-center justify-center bg-white text-gray-600 hover:bg-gray-200 rounded-full shadow-md border border-gray-300 focus:outline-none focus:ring-2 focus:ring-blue-500';
  closeBtn.setAttribute('aria-label', t.closeLabel);
  
  // Create SVG icon for close button.
  const svgns = "http://www.w3.org/2000/svg";
  const closeIcon = document.createElementNS(svgns, 'svg');
  closeIcon.setAttribute('class', 'w-4 h-4');
  closeIcon.setAttribute('fill', 'none');
  closeIcon.setAttribute('stroke', 'currentColor');
  closeIcon.setAttribute('stroke-width', '2');
  closeIcon.setAttribute('viewBox', '0 0 24 24');
  closeIcon.innerHTML = '<path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"></path>';
  closeBtn.appendChild(closeIcon);
  modalDiv.appendChild(closeBtn);

  // Create iframe container and iframe with lazy loading.
  const iframeDiv = document.createElement('div');
  iframeDiv.className = 'relative pt-[56.25%]'; // Maintain 16:9 aspect ratio.
  const iframe = document.createElement('iframe');
  iframe.id = 'video-frame';
  iframe.className = 'absolute top-0 left-0 w-full h-full';
  iframe.setAttribute('data-src', videoUrl); // Store URL in data-src
  iframe.src = videoUrl; // Load immediately when modal opens
  iframe.setAttribute('title', t.videoTitle);
  iframe.setAttribute('loading', 'lazy'); // Lazy loading
  iframe.setAttribute('allow', 'accelerometer; autoplay; encrypted-media; gyroscope; picture-in-picture');
  iframe.setAttribute('allowfullscreen', '');
  iframeDiv.appendChild(iframe);
  modalDiv.appendChild(iframeDiv);

  // Append modal to target container.
  target.appendChild(modalDiv);

  // Animate modal in.
  requestAnimationFrame(() => {
    modalDiv.classList.remove('opacity-0');
    modalDiv.classList.add('opacity-100');
    closeBtn.focus();
  });

  // Define event handlers.
  const handleEscape = (e) => {
    if (e.key === 'Escape') {
      closeModal();
    }
  };
  const handleCloseClick = () => {
    closeModal();
  };

  document.addEventListener('keydown', handleEscape);
  closeBtn.addEventListener('click', handleCloseClick);

  // Cleanup and close modal function.
  function closeModal() {
    iframe.src = ''; // Unload iframe
    modalDiv.classList.remove('opacity-100');
    modalDiv.classList.add('opacity-0');
    setTimeout(() => {
      if (target.contains(modalDiv)) {
        target.removeChild(modalDiv);
      }
      if (!target.classList.contains('hidden')) {
        target.classList.add('hidden');
      }
      if (previouslyFocused && typeof previouslyFocused.focus === 'function') {
        previouslyFocused.focus();
      }
    }, 200);
    document.removeEventListener('keydown', handleEscape);
    closeBtn.removeEventListener('click', handleCloseClick);
  }

  return closeModal;
}