import '/src/styles/main.css';
import '/src/styles/fonts.css';

import {
  renderMetaNav,
  renderMainNav,
  renderMobileMenu,
  renderMobileSearch
} from './components/Navigation.js';
import { renderVideoModal } from './components/VideoModal.js';
import { renderFooter } from './components/Footer.js';
import { renderHero } from './components/Hero.js';
import { ComponentLoader } from './components/ComponentLoader.js';
import { events } from './utils/EventManager.js';

// Component ID configuration
const componentIds = {
  metaNav: 'meta-nav',
  mainNav: 'main-nav',
  mobileMenu: 'mobile-menu',
  mobileSearch: 'mobile-search',
  hero: 'hero',
  footer: 'footer',
  videoModal: 'video-modal',
};

// Initialize the component loader
const loader = new ComponentLoader();
loader.register('metaNav', {
  render: renderMetaNav,
  cleanup: (target) => {
    if (target.__cleanupMetaNav) {
      target.__cleanupMetaNav();
      delete target.__cleanupMetaNav;
    }
  }
});
loader.register('mainNav', { render: renderMainNav });
loader.register('mobileMenu', { render: renderMobileMenu });
loader.register('mobileSearch', { render: renderMobileSearch });
loader.register('videoModal', { render: renderVideoModal });
loader.register('hero', { render: renderHero });
loader.register('footer', { render: renderFooter });

// Routing for extensibility
const routes = {
  '#home': () => {
    loader.render('hero', componentIds.hero);
    loader.render('footer', componentIds.footer);
  },
  '#product': () => {
    document.querySelector('main').innerHTML =
      '<section class="max-w-7xl mx-auto px-4 py-8"><h1 class="hero-title">Product Page</h1></section>';
    loader.render('footer', componentIds.footer);
  },
};

// Fallback for invalid routes
const fallbackRoute = () => {
  window.location.hash = '#home';
  routes['#home']();
};

/**
 * Renders the page based on the current hash
 */
function renderPage() {
  const hash = window.location.hash || '#home';
  const route = routes[hash];
  if (route) {
    route();
  } else {
    console.warn(`Route ${hash} not found. Redirecting to home.`);
    fallbackRoute();
  }
}

/**
 * Renders shared components on initial load
 */
function initializeComponents() {
  loader.render('metaNav', componentIds.metaNav);
  loader.render('mainNav', componentIds.mainNav);
  loader.render('mobileMenu', componentIds.mobileMenu);
  loader.render('mobileSearch', componentIds.mobileSearch);
}

// Initialize on DOM load and apply fade-in
document.addEventListener('DOMContentLoaded', () => {
  initializeComponents();
  renderPage();

  // Trigger fade-in after components are rendered
  requestAnimationFrame(() => {
    document.body.classList.add('loaded');
  });

  // Global event delegation
  events.on('click', document.body, (e) => {
    const action = e.target.closest('[data-action]')?.dataset.action;
    if (!action) return;

    switch (action) {
      case 'toggle-lang':
        events.emit('toggleDropdown', { id: 'lang-dropdown', trigger: e.target.closest('button'), focusId: null });
        break;
      case 'toggle-search':
        if (window.innerWidth < 768) {
          events.emit('toggleOverlay', { id: componentIds.mobileSearch, iframeId: null, open: true });
        } else {
          events.emit('toggleDropdown', { id: 'search-field', trigger: e.target.closest('button'), focusId: 'search-input' });
        }
        break;
      case 'toggle-mobile':
        events.emit('toggleOverlay', { id: componentIds.mobileMenu, iframeId: null });
        break;
      case 'close-search':
        events.emit('toggleOverlay', { id: componentIds.mobileSearch, iframeId: null, open: false });
        break;
      case 'open-video': {
        const videoContainer = document.getElementById(componentIds.videoModal);
        const videoUrl = e.target.closest('[data-action="open-video"]').dataset.videoUrl || 'https://www.youtube.com/embed/8KGN6e0OnDY?modestbranding=1&rel=0&autoplay=1';
        if (videoContainer) {
          videoContainer.innerHTML = '';
          renderVideoModal(videoContainer, 'en', videoUrl);
          videoContainer.classList.remove('hidden');
        }
        break;
      }
      default:
        break;
    }

    events.emit('clickOutside', e);
  });

  // Global keydown listener for shortcuts
  events.on('keydown', document, (e) => {
    if (e.metaKey && e.key.toLowerCase() === 'k') {
      e.preventDefault();
      events.emit('toggleOverlay', { id: componentIds.mobileSearch, iframeId: null, open: true });
    }
  });

  // Update route on hash change
  window.addEventListener('hashchange', renderPage);
});