/**
 * Manages event listeners and custom events.
 */
import {
  toggleDropdown,
  toggleOverlay,
  handleClickOutside,
} from './domUtils.js';

export class EventManager {
  constructor() {
    this.listeners = new Map();
    this.DEBUG = import.meta.env.MODE !== 'production';
  }

  /**
   * Adds an event listener to an element.
   * @param {string} eventName - The event name (e.g., 'click', 'keydown').
   * @param {HTMLElement} element - The element to attach the listener to.
   * @param {Function} callback - The callback function.
   */
  on(eventName, element, callback) {
    if (!this.listeners.has(eventName)) {
      this.listeners.set(eventName, new Map());
    }
    const eventListeners = this.listeners.get(eventName);
    eventListeners.set(callback, callback);
    element.addEventListener(eventName, callback);
  }

  /**
   * Emits a custom event with data.
   * @param {string} eventName - The custom event name.
   * @param {Object} data - The data to pass to listeners.
   */
  emit(eventName, data) {
    const eventListeners = this.listeners.get(eventName) || new Map();
    eventListeners.forEach((callback) => callback(data));
  }

  /**
   * Removes an event listener.
   * @param {string} eventName - The event name.
   * @param {HTMLElement} element - The element to remove the listener from.
   * @param {Function} callback - The callback function to remove.
   */
  off(eventName, element, callback) {
    const eventListeners = this.listeners.get(eventName);
    if (eventListeners && eventListeners.has(callback)) {
      element.removeEventListener(eventName, callback);
      eventListeners.delete(callback);
    }
  }
}

// Export a singleton instance.
export const events = new EventManager();

// Register custom events from domUtils.
events.on('toggleDropdown', document, toggleDropdown);
events.on('toggleOverlay', document, toggleOverlay);
events.on('clickOutside', document, handleClickOutside);
