import { trapFocus } from './focusTrap.js';

const DEBUG = (import.meta.env?.MODE ?? 'production') !== 'production';

let focusedElementBeforeModal;

/**
 * Stores the currently focused element.
 */
function storeFocus() {
  focusedElementBeforeModal = document.activeElement;
}

/**
 * Restores focus to the previously focused element.
 */
function restoreFocus() {
  if (focusedElementBeforeModal && typeof focusedElementBeforeModal.focus === 'function') {
    focusedElementBeforeModal.focus();
  }
}

/**
 * Toggles a dropdown with smooth transitions.
 * @param {Object} data - Contains id, trigger, and optional focusId.
 * @returns {Function|null} Cleanup function if opened.
 */
export function toggleDropdown({ id, trigger, focusId }) {
  try {
    const dropdown = document.getElementById(id);
    if (!dropdown) {
      if (DEBUG) console.warn(`Dropdown with ID ${id} not found.`);
      return null;
    }
    const isOpen = !dropdown.classList.contains('scale-0');
    if (isOpen) {
      dropdown.classList.remove('scale-100', 'opacity-100');
      dropdown.classList.add('scale-0', 'opacity-0', 'pointer-events-none');
      trigger.setAttribute('aria-expanded', 'false');
      setTimeout(() => {
        dropdown.classList.add('hidden');
      }, 200);
    } else {
      dropdown.classList.remove('hidden', 'scale-0', 'opacity-0', 'pointer-events-none');
      dropdown.classList.add('scale-100', 'opacity-100');
      trigger.setAttribute('aria-expanded', 'true');
      if (focusId) {
        setTimeout(() => {
          const focusEl = document.getElementById(focusId);
          if (focusEl) focusEl.focus();
          else if (DEBUG) console.warn(`Focus element with ID ${focusId} not found.`);
        }, 200);
      }
      trapFocus(dropdown);
    }
    return null;
  } catch (error) {
    if (DEBUG) console.error(`Error in toggleDropdown: ${error.message}`);
    return null;
  }
}

/**
 * Toggles an overlay with a smooth transition.
 * @param {Object} data - Contains id, iframeId, and optional open flag.
 */
export function toggleOverlay({ id, iframeId, open }) {
  try {
    const overlay = document.getElementById(id);
    if (!overlay) {
      if (DEBUG) console.warn(`Overlay with ID ${id} not found.`);
      return;
    }
    const isOpen = !overlay.classList.contains('hidden');

    if (id === 'mobile-menu') {
      if (open !== undefined ? open : !isOpen) {
        overlay.classList.remove('hidden', 'translate-x-full');
        overlay.classList.add('translate-x-0');
      } else {
        overlay.classList.remove('translate-x-0');
        overlay.classList.add('translate-x-full');
        setTimeout(() => {
          overlay.classList.add('hidden');
        }, 200);
      }
      return;
    }

    if (open !== undefined ? open : !isOpen) {
      overlay.classList.remove('hidden', 'opacity-0');
      overlay.classList.add('opacity-100', 'flex');
      storeFocus();
      const cleanup = trapFocus(overlay);
      overlay.addEventListener('close', () => {
        cleanup();
        restoreFocus();
      }, { once: true });
      if (iframeId) {
        const iframe = document.getElementById(iframeId);
        if (iframe) {
          const src = iframe.getAttribute('data-src') || '';
          iframe.src = src;
        } else if (DEBUG) console.warn(`Iframe with ID ${iframeId} not found.`);
      }
      if (id === 'video-modal') {
        const content = overlay.querySelector('div');
        if (content) {
          setTimeout(() => content.classList.replace('opacity-0', 'opacity-100'), 50);
        } else if (DEBUG) console.warn('Video modal content not found.');
      }
    } else {
      overlay.classList.remove('opacity-100', 'flex');
      overlay.classList.add('opacity-0');
      setTimeout(() => {
        overlay.classList.add('hidden');
        if (iframeId) {
          const iframe = document.getElementById(iframeId);
          if (iframe) iframe.src = '';
        }
        overlay.dispatchEvent(new Event('close'));
      }, 200);
    }
  } catch (error) {
    if (DEBUG) console.error(`Error in toggleOverlay: ${error.message}`);
  }
}

/**
 * Handles click-outside behavior for dropdowns and overlays.
 * @param {Event} e - The click event.
 */
export function handleClickOutside(e) {
  try {
    const langDropdown = document.getElementById('lang-dropdown');
    const searchField = document.getElementById('search-field');
    const videoModal = document.getElementById('video-modal');
    const mobileSearch = document.getElementById('mobile-search');
    const mobileMenu = document.getElementById('mobile-menu');

    if (langDropdown && !e.target.closest('#lang-container') && !langDropdown.classList.contains('scale-0')) {
      toggleDropdown({
        id: 'lang-dropdown',
        trigger: document.querySelector('[data-action="toggle-lang"]'),
        focusId: null,
      });
    }
    if (searchField && !e.target.closest('#search-container') && !searchField.classList.contains('scale-0')) {
      toggleDropdown({
        id: 'search-field',
        trigger: document.getElementById('search-btn'),
        focusId: null,
      });
    }
    if (videoModal && e.target.closest('#video-modal') && !e.target.closest('#video-modal > div')) {
      toggleOverlay({ id: 'video-modal', iframeId: 'video-frame', open: false });
    }
    if (mobileSearch && e.target.closest('#mobile-search') && !e.target.closest('#mobile-search > div')) {
      toggleOverlay({ id: 'mobile-search', iframeId: null, open: false });
    }
    if (mobileMenu && !e.target.closest('#mobile-menu') && !e.target.closest('[data-action="toggle-mobile"]')) {
      toggleOverlay({ id: 'mobile-menu', iframeId: null, open: false });
    }
  } catch (error) {
    if (DEBUG) console.error(`Error in handleClickOutside: ${error.message}`);
  }
}