/**
 * Traps focus within a container for accessibility.
 * @param {HTMLElement} container - The container to trap focus within.
 * @returns {Function} Cleanup function to remove the event listener.
 */
export function trapFocus(container) {
  const focusable = container.querySelectorAll('a[href], button, input, textarea, select, [tabindex="0"]');
  if (!focusable.length) return () => {};
  const first = focusable[0];
  const last = focusable[focusable.length - 1];

  const handleKey = (e) => {
    if (e.key === 'Tab') {
      if (e.shiftKey && document.activeElement === first) {
        e.preventDefault();
        last.focus();
      } else if (!e.shiftKey && document.activeElement === last) {
        e.preventDefault();
        first.focus();
      }
    } else if (e.key === 'Escape') {
      container.dispatchEvent(new Event('close'));
    }
  };

  container.addEventListener('keydown', handleKey);
  return () => container.removeEventListener('keydown', handleKey);
}
